const test = require('node:test');
const assert = require('node:assert/strict');

const {
  AnalyticsEventRegistry,
  FALLBACK_REGISTRY,
  validateAndNormalizeConfig,
  normalizeRemoteMetadata
} = require('../analytics/registry');

test('fallback registry blocks unknown events by default', () => {
  const registry = new AnalyticsEventRegistry({ fallbackConfig: FALLBACK_REGISTRY });
  const evaluation = registry.evaluateEvent('unregistered_event', { foo: 'bar' });
  assert.strictEqual(evaluation.allow, false);
  assert.strictEqual(evaluation.reason, 'fallback_registry_blocked');
  assert.deepStrictEqual(evaluation.properties, {});
});

test('custom registry allows configured events and applies transforms', () => {
  const registry = new AnalyticsEventRegistry({ fallbackConfig: FALLBACK_REGISTRY });
  const customConfig = {
    version: 1,
    remote: {
      url: 'https://example.com/events.json',
      timeoutMs: 5000
    },
    defaults: {
      allow: false,
      propertyAllowlist: ['source'],
      propertyTransforms: {}
    },
    events: {
      example_event: {
        allow: true,
        propertyAllowlist: ['value', 'notes'],
        propertyTransforms: {
          value: { type: 'truncate', maxLength: 4 },
          notes: { type: 'enum', values: ['a', 'b', 'c'] }
        }
      }
    }
  };

  const updated = registry.update(customConfig, { source: 'test' });
  assert.ok(updated, 'registry should accept valid config');

  const evaluation = registry.evaluateEvent('example_event', {
    value: '123456',
    notes: 'z',
    source: 'renderer',
    ignored: 'nope'
  });

  assert.strictEqual(evaluation.allow, true);
  assert.strictEqual(evaluation.properties.value, '1234');
  assert.ok(!Object.prototype.hasOwnProperty.call(evaluation.properties, 'notes'), 'invalid enum values are dropped');
  assert.strictEqual(evaluation.droppedProperties.notes, 'value_not_in_enum');
  assert.strictEqual(evaluation.droppedProperties.ignored, 'property_not_allowlisted');
});

test('validateAndNormalizeConfig rejects incompatible versions', () => {
  const { errors } = validateAndNormalizeConfig({ version: 2 }, { source: 'test' });
  assert.ok(Array.isArray(errors));
  assert.notStrictEqual(errors.length, 0);
});

test('normalizeRemoteMetadata fills defaults and sanitises input', () => {
  const remote = normalizeRemoteMetadata(
    { url: ' https://example.com ', timeoutMs: '2000' },
    { fallback: FALLBACK_REGISTRY.remote }
  );

  assert.strictEqual(remote.url, 'https://example.com');
  assert.strictEqual(remote.timeoutMs, 2000);
});


